/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *    https://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 */

package grails.async.web

import groovy.transform.CompileStatic

import jakarta.servlet.AsyncContext
import jakarta.servlet.http.HttpServletRequest

import org.springframework.web.context.request.RequestContextHolder
import org.springframework.web.context.request.async.AsyncWebRequest
import org.springframework.web.context.request.async.WebAsyncManager
import org.springframework.web.context.request.async.WebAsyncUtils

import org.grails.plugins.web.async.GrailsAsyncContext
import org.grails.web.servlet.mvc.GrailsWebRequest
import org.grails.web.util.GrailsApplicationAttributes

/**
 * Exposes a startAsync() method for access to the Servlet 3.x API
 *
 * @author Graeme Rocher
 * @since 3.3
 */
@CompileStatic
trait AsyncController {

    /**
     * Raw access to the Servlet 3.0 startAsync method
     *
     * @return a new {@link javax.servlet.AsyncContext}
     */
    AsyncContext startAsync() {
        GrailsWebRequest webRequest = (GrailsWebRequest) RequestContextHolder.currentRequestAttributes()

        HttpServletRequest request = webRequest.currentRequest
        WebAsyncManager asyncManager = WebAsyncUtils.getAsyncManager(request)

        AsyncWebRequest asyncWebRequest = new AsyncGrailsWebRequest(request, webRequest.currentResponse, webRequest.servletContext)
        asyncManager.setAsyncWebRequest(asyncWebRequest)

        asyncWebRequest.startAsync()
        request.setAttribute(GrailsApplicationAttributes.ASYNC_STARTED, true)
        new GrailsAsyncContext(asyncWebRequest.asyncContext, webRequest)
    }
}
