/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.juneau.annotation;

import static java.lang.annotation.ElementType.*;
import static java.lang.annotation.RetentionPolicy.*;
import static org.apache.juneau.commons.utils.CollectionUtils.*;

import java.lang.annotation.*;

import org.apache.juneau.*;
import org.apache.juneau.commons.annotation.*;
import org.apache.juneau.commons.reflect.*;
import org.apache.juneau.svl.*;
import org.apache.juneau.swap.*;

/**
 * Utility classes and methods for the {@link Bean @Bean} annotation.
 *
 */
public class BeanAnnotation {
	/**
	 * Applies targeted {@link Bean} annotations to a {@link org.apache.juneau.BeanContext.Builder}.
	 */
	public static class Applier extends AnnotationApplier<Bean,BeanContext.Builder> {

		/**
		 * Constructor.
		 *
		 * @param vr The resolver for resolving values in annotations.
		 */
		public Applier(VarResolverSession vr) {
			super(Bean.class, BeanContext.Builder.class, vr);
		}

		@Override
		public void apply(AnnotationInfo<Bean> ai, BeanContext.Builder b) {
			Bean a = ai.inner();
			if (isEmptyArray(a.on()) && isEmptyArray(a.onClass()))
				return;
			b.annotations(copy(a, vr()));
		}
	}

	/**
	 * A collection of {@link Bean @Bean annotations}.
	 */
	@Documented
	@Target({ METHOD, TYPE })
	@Retention(RUNTIME)
	@Inherited
	public static @interface Array {

		/**
		 * The child annotations.
		 *
		 * @return The annotation value.
		 */
		Bean[] value();
	}

	/**
	 * Builder class.
	 *
	 * <h5 class='section'>See Also:</h5><ul>
	 * 	<li class='jm'>{@link org.apache.juneau.BeanContext.Builder#annotations(Annotation...)}
	 * </ul>
	 */
	public static class Builder extends AppliedAnnotationObject.BuilderT {

		private String[] description = {};
		private Class<?>[] dictionary = new Class[0];
		private Class<?> implClass = void.class;
		private Class<?> interfaceClass = void.class;
		private Class<?> stopClass = void.class;
		private Class<? extends BeanInterceptor<?>> interceptor = BeanInterceptor.Void.class;
		private Class<? extends PropertyNamer> propertyNamer = BasicPropertyNamer.class;
		private String example = "", excludeProperties = "", p = "", properties = "", readOnlyProperties = "", ro = "", typeName = "", typePropertyName = "", wo = "", writeOnlyProperties = "",
			xp = "";
		private boolean findFluentSetters, sort;

		/**
		 * Constructor.
		 */
		protected Builder() {
			super(Bean.class);
		}

		/**
		 * Instantiates a new {@link Bean @Bean} object initialized with this builder.
		 *
		 * @return A new {@link Bean @Bean} object.
		 */
		public Bean build() {
			return new Object(this);
		}

		/**
		 * Sets the description property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder description(String...value) {
			description = value;
			return this;
		}

		/**
		 * Sets the {@link Bean#dictionary()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder dictionary(Class<?>...value) {
			dictionary = value;
			return this;
		}

		/**
		 * Sets the {@link Bean#example()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder example(String value) {
			example = value;
			return this;
		}

		/**
		 * Sets the {@link Bean#excludeProperties()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder excludeProperties(String value) {
			excludeProperties = value;
			return this;
		}

		/**
		 * Sets the {@link Bean#findFluentSetters()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder findFluentSetters(boolean value) {
			findFluentSetters = value;
			return this;
		}

		/**
		 * Sets the {@link Bean#implClass()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder implClass(Class<?> value) {
			implClass = value;
			return this;
		}

		/**
		 * Sets the {@link Bean#interceptor()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder interceptor(Class<? extends BeanInterceptor<?>> value) {
			interceptor = value;
			return this;
		}

		/**
		 * Sets the {@link Bean#interfaceClass()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder interfaceClass(Class<?> value) {
			interfaceClass = value;
			return this;
		}

		/**
		 * Sets the {@link Bean#p()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder p(String value) {
			p = value;
			return this;
		}

		/**
		 * Sets the {@link Bean#properties()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder properties(String value) {
			properties = value;
			return this;
		}

		/**
		 * Sets the {@link Bean#propertyNamer()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder propertyNamer(Class<? extends PropertyNamer> value) {
			propertyNamer = value;
			return this;
		}

		/**
		 * Sets the {@link Bean#readOnlyProperties()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder readOnlyProperties(String value) {
			readOnlyProperties = value;
			return this;
		}

		/**
		 * Sets the {@link Bean#ro()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder ro(String value) {
			ro = value;
			return this;
		}

		/**
		 * Sets the {@link Bean#sort()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder sort(boolean value) {
			sort = value;
			return this;
		}

		/**
		 * Sets the {@link Bean#stopClass()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder stopClass(Class<?> value) {
			stopClass = value;
			return this;
		}

		/**
		 * Sets the {@link Bean#typeName()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder typeName(String value) {
			typeName = value;
			return this;
		}

		/**
		 * Sets the {@link Bean#typePropertyName()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder typePropertyName(String value) {
			typePropertyName = value;
			return this;
		}

		/**
		 * Sets the{@link Bean#wo()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder wo(String value) {
			wo = value;
			return this;
		}

		/**
		 * Sets the{@link Bean#writeOnlyProperties()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder writeOnlyProperties(String value) {
			writeOnlyProperties = value;
			return this;
		}

		/**
		 * Sets the {@link Bean#xp()} property on this annotation.
		 *
		 * @param value The new value for this property.
		 * @return This object.
		 */
		public Builder xp(String value) {
			xp = value;
			return this;
		}

		@Override /* Overridden from AppliedAnnotationObject.Builder */
		public Builder on(String...value) {
			super.on(value);
			return this;
		}

		@Override /* Overridden from AppliedAnnotationObject.BuilderT */
		public Builder on(Class<?>...value) {
			super.on(value);
			return this;
		}

		@Override /* Overridden from AppliedOnClassAnnotationObject.Builder */
		public Builder onClass(Class<?>...value) {
			super.onClass(value);
			return this;
		}

		@Override /* Overridden from AppliedAnnotationObject.BuilderT */
		public Builder on(ClassInfo...value) {
			super.on(value);
			return this;
		}

		@Override /* Overridden from AppliedAnnotationObject.BuilderT */
		public Builder onClass(ClassInfo...value) {
			super.onClass(value);
			return this;
		}

	}

	private static class Object extends AppliedOnClassAnnotationObject implements Bean {

		private final String[] description;
		private final boolean findFluentSetters, sort;
		private final Class<? extends BeanInterceptor<?>> interceptor;
		private final Class<? extends PropertyNamer> propertyNamer;
		private final Class<?> implClass, interfaceClass, stopClass;
		private final Class<?>[] dictionary;
		private final String example, excludeProperties, p, properties, readOnlyProperties, ro, typeName, typePropertyName, wo, writeOnlyProperties, xp;

		Object(BeanAnnotation.Builder b) {
			super(b);
			description = copyOf(b.description);
			dictionary = copyOf(b.dictionary);
			example = b.example;
			excludeProperties = b.excludeProperties;
			findFluentSetters = b.findFluentSetters;
			implClass = b.implClass;
			interceptor = b.interceptor;
			interfaceClass = b.interfaceClass;
			p = b.p;
			properties = b.properties;
			propertyNamer = b.propertyNamer;
			readOnlyProperties = b.readOnlyProperties;
			ro = b.ro;
			sort = b.sort;
			stopClass = b.stopClass;
			typeName = b.typeName;
			typePropertyName = b.typePropertyName;
			wo = b.wo;
			writeOnlyProperties = b.writeOnlyProperties;
			xp = b.xp;
		}

		@Override /* Overridden from Bean */
		public Class<?>[] dictionary() {
			return dictionary;
		}

		@Override /* Overridden from Bean */
		public String example() {
			return example;
		}

		@Override /* Overridden from Bean */
		public String excludeProperties() {
			return excludeProperties;
		}

		@Override /* Overridden from Bean */
		public boolean findFluentSetters() {
			return findFluentSetters;
		}

		@Override /* Overridden from Bean */
		public Class<?> implClass() {
			return implClass;
		}

		@Override /* Overridden from Bean */
		public Class<? extends BeanInterceptor<?>> interceptor() {
			return interceptor;
		}

		@Override /* Overridden from Bean */
		public Class<?> interfaceClass() {
			return interfaceClass;
		}

		@Override /* Overridden from Bean */
		public String p() {
			return p;
		}

		@Override /* Overridden from Bean */
		public String properties() {
			return properties;
		}

		@Override /* Overridden from Bean */
		public Class<? extends PropertyNamer> propertyNamer() {
			return propertyNamer;
		}

		@Override /* Overridden from Bean */
		public String readOnlyProperties() {
			return readOnlyProperties;
		}

		@Override /* Overridden from Bean */
		public String ro() {
			return ro;
		}

		@Override /* Overridden from Bean */
		public boolean sort() {
			return sort;
		}

		@Override /* Overridden from Bean */
		public Class<?> stopClass() {
			return stopClass;
		}

		@Override /* Overridden from Bean */
		public String typeName() {
			return typeName;
		}

		@Override /* Overridden from Bean */
		public String typePropertyName() {
			return typePropertyName;
		}

		@Override /* Overridden from Bean */
		public String wo() {
			return wo;
		}

		@Override /* Overridden from Bean */
		public String writeOnlyProperties() {
			return writeOnlyProperties;
		}

		@Override /* Overridden from Bean */
		public String xp() {
			return xp;
		}

		@Override /* Overridden from Bean */
		public String[] description() {
			return description;
		}
	}

	/** Default value */
	public static final Bean DEFAULT = create().build();

	/**
	 * Creates a copy of the specified annotation.
	 *
	 * @param a The annotation to copy.
	 * @param r The var resolver for resolving any variables.
	 * @return A copy of the specified annotation.
	 */
	public static Bean copy(Bean a, VarResolverSession r) {
		// @formatter:off
		return
			create()
			.dictionary(a.dictionary())
			.example(r.resolve(a.example()))
			.excludeProperties(r.resolve(a.excludeProperties()))
			.findFluentSetters(a.findFluentSetters())
			.implClass(a.implClass())
			.interceptor(a.interceptor())
			.interfaceClass(a.interfaceClass())
			.on(r.resolve(a.on()))
			.onClass(a.onClass())
			.p(r.resolve(a.p()))
			.properties(r.resolve(a.properties()))
			.propertyNamer(a.propertyNamer())
			.readOnlyProperties(r.resolve(a.readOnlyProperties()))
			.ro(r.resolve(a.ro()))
			.sort(a.sort())
			.stopClass(a.stopClass())
			.typeName(r.resolve(a.typeName()))
			.typePropertyName(r.resolve(a.typePropertyName()))
			.wo(r.resolve(a.wo()))
			.writeOnlyProperties(r.resolve(a.writeOnlyProperties()))
			.xp(r.resolve(a.xp()))
			.build();
		// @formatter:on
	}

	/**
	 * Instantiates a new builder for this class.
	 *
	 * @return A new builder object.
	 */
	public static Builder create() {
		return new Builder();
	}

	/**
	 * Instantiates a new builder for this class.
	 *
	 * @param on The targets this annotation applies to.
	 * @return A new builder object.
	 */
	public static Builder create(Class<?>...on) {
		return create().on(on);
	}

	/**
	 * Instantiates a new builder for this class.
	 *
	 * @param on The targets this annotation applies to.
	 * @return A new builder object.
	 */
	public static Builder create(String...on) {
		return create().on(on);
	}
}